import { NextRequest, NextResponse } from 'next/server';
import Database from 'better-sqlite3';
import path from 'path';

function getDb() {
    const dbPath = path.join(process.cwd(), 'comprehensive.db');
    return new Database(dbPath);
}

// GET /api/events/[id] - קבלת אירוע ספציפי
export async function GET(request: NextRequest, { params }: { params: Promise<{ id: string }> }) {
    const { id } = await params;
    const db = getDb();
    try {
        const event = db.prepare(`
            SELECT 
                e.*,
                c.name as customer_name,
                c.company as customer_company,
                c.email as customer_email,
                c.phone as customer_phone,
                d.name as domain_name,
                p.name as project_title
            FROM events e
            LEFT JOIN customers c ON e.customer_id = c.id
            LEFT JOIN domains d ON e.domain_id = d.id
            LEFT JOIN projects p ON e.project_id = p.id
            WHERE e.id = ?
        `).get(id) as unknown;

        if (!event) {
            return NextResponse.json({ error: 'אירוע לא נמצא' }, { status: 404 });
        }

        // קבלת משימות קשורות (אופציונלי - אם טבלת tasks קיימת)
        let relatedTasks: any[] = [];
        try {
            // בדיקה אם טבלת tasks קיימת
            const tableExists = db.prepare(`SELECT name FROM sqlite_master WHERE type='table' AND name='tasks'`).get();
            if (tableExists) {
                relatedTasks = db.prepare(`
                    SELECT 
                        t.id, t.title, t.status, t.priority, t.due_date,
                        t.assigned_to
                    FROM tasks t
                    WHERE t.event_id = ?
                    ORDER BY t.due_date ASC
                `).all(id) as unknown as any[];
            }
        } catch (e) {
            console.log('No tasks table found, skipping related tasks');
        }

        return NextResponse.json({
            ...event,
            related_tasks: relatedTasks
        });

    } catch (e: unknown) {
        console.error('Get event error:', e);
        return NextResponse.json({ error: 'שגיאה בטעינת אירוע: ' + (e as Error).message }, { status: 500 });
    } finally {
        db.close();
    }
}

// PUT /api/events/[id] - עדכון אירוע
export async function PUT(request: NextRequest, { params }: { params: Promise<{ id: string }> }) {
    const { id } = await params;
    const db = getDb();
    try {
        const body = await request.json();
        console.log('🔄 Updating event:', id);
        console.log('📝 Request body:', body);
        console.log('🏷️ Tags in request:', body.tags);

        // בדיקה שהאירוע קיים
        const existingEvent = db.prepare('SELECT id FROM events WHERE id = ?').get(id) as unknown;
        if (!existingEvent) {
            return NextResponse.json({ error: 'אירוע לא נמצא' }, { status: 404 });
        }

        // וולידציה: אם יש customer_id, לוודא שהוא קיים
        if (body.customer_id) {
            const customer = db.prepare('SELECT id FROM customers WHERE id = ?').get(body.customer_id) as unknown;
            if (!customer) {
                return NextResponse.json(
                    { error: 'לקוח לא נמצא' },
                    { status: 400 }
                );
            }
        }

        // וולידציה: אם יש domain_id, לוודא שהוא קיים ושייך ללקוח
        if (body.domain_id && body.customer_id) {
            const domain = db.prepare('SELECT customer_id FROM domains WHERE id = ?').get(body.domain_id) as unknown;
            if (!domain) {
                return NextResponse.json(
                    { error: 'דומיין לא נמצא' },
                    { status: 400 }
                );
            }
            if ((domain as { customer_id: string }).customer_id !== body.customer_id) {
                return NextResponse.json(
                    { error: 'הדומיין לא שייך ללקוח שנבחר' },
                    { status: 400 }
                );
            }
        }

        // וולידציה: אם יש project_id, לוודא שהוא קיים ושייך ללקוח
        if (body.project_id && body.customer_id) {
            const project = db.prepare('SELECT customer_id FROM projects WHERE id = ?').get(body.project_id) as unknown;
            if (!project) {
                return NextResponse.json(
                    { error: 'פרויקט לא נמצא' },
                    { status: 400 }
                );
            }
            if ((project as { customer_id: string }).customer_id !== body.customer_id) {
                return NextResponse.json(
                    { error: 'הפרויקט לא שייך ללקוח שנבחר' },
                    { status: 400 }
                );
            }
        }

        const updateQuery = `
            UPDATE events SET
                title = COALESCE(?, title),
                description = COALESCE(?, description),
                start_datetime = COALESCE(?, start_datetime),
                end_datetime = COALESCE(?, end_datetime),
                location = COALESCE(?, location),
                customer_id = COALESCE(?, customer_id),
                domain_id = COALESCE(?, domain_id),
                project_id = COALESCE(?, project_id),
                quote_id = COALESCE(?, quote_id),
                event_type = COALESCE(?, event_type),
                status = COALESCE(?, status),
                attendees = COALESCE(?, attendees),
                reminder_minutes = COALESCE(?, reminder_minutes),
                tags = COALESCE(?, tags),
                updated_at = CURRENT_TIMESTAMP
            WHERE id = ?
        `;

        const tagsJson = body.tags ? JSON.stringify(body.tags) : null;
        console.log('🏷️ Tags JSON to save:', tagsJson);

        db.prepare(updateQuery).run(
            body.title || null,
            body.description || null,
            body.start_datetime || null,
            body.end_datetime || null,
            body.location || null,
            body.customer_id || null,
            body.domain_id || null,
            body.project_id || null,
            body.quote_id || null,
            body.event_type || null,
            body.status || null,
            body.attendees ? JSON.stringify(body.attendees) : null,
            body.reminder_minutes || null,
            tagsJson,
            id
        );

        console.log('✅ Event updated successfully');

        // קבלת האירוע המעודכן
        const updatedEvent = db.prepare(`
            SELECT 
                e.*,
                c.name as customer_name,
                c.company as customer_company,
                c.email as customer_email,
                c.phone as customer_phone,
                d.name as domain_name,
                p.name as project_title
            FROM events e
            LEFT JOIN customers c ON e.customer_id = c.id
            LEFT JOIN domains d ON e.domain_id = d.id
            LEFT JOIN projects p ON e.project_id = p.id
            WHERE e.id = ?
        `).get(id) as unknown;

        return NextResponse.json(updatedEvent);

    } catch (e: unknown) {
        console.error('Update event error:', e);
        return NextResponse.json({ error: 'שגיאה בעדכון אירוע: ' + (e as Error).message }, { status: 500 });
    } finally {
        db.close();
    }
}

// DELETE /api/events/[id] - מחיקת אירוע
export async function DELETE(request: NextRequest, { params }: { params: Promise<{ id: string }> }) {
    const { id } = await params;
    const db = getDb();
    try {
        // בדיקה שהאירוע קיים
        const existingEvent = db.prepare('SELECT id FROM events WHERE id = ?').get(id) as unknown;
        if (!existingEvent) {
            return NextResponse.json({ error: 'אירוע לא נמצא' }, { status: 404 });
        }

        // מחיקת האירוע (CASCADE ימחק גם משימות קשורות)
        db.prepare('DELETE FROM events WHERE id = ?').run(id);

        return NextResponse.json({ message: 'אירוע נמחק בהצלחה' });

    } catch (e: unknown) {
        console.error('Delete event error:', e);
        return NextResponse.json({ error: 'שגיאה במחיקת אירוע: ' + (e as Error).message }, { status: 500 });
    } finally {
        db.close();
    }
}
